// PCXF.CPP - PCX graphic functions
//~~~~~~~~~~~~~~~~~~~~~~~~~~INCLUDES~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#include <stdio.h>
#include "pcxf.h"
#include <stdlib.h>
#include <math.h>
#include <string.h>
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~GLOBALS~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
pcx *letters[95]; //create array of pointers to pcx structs.
int fontsLoaded = 0; //if fonts have been loaded or not.
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~CONSTANTS~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#define M_PI 3.14159265358979323846

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#define RGB16BIT555(r, g, b) ((b & 31) + ((g & 31) << 5) + ((r & 31) << 10))
#define RGB16BIT565(r, g, b) ((b & 31) + ((g & 63) << 5) + ((r & 31) << 11))
#define RGB24BIT(r, g, b) ((b) + ((g) << 8) + ((r) << 16))
#define RGB32BIT(r, g, b, a) ((b) + ((g) << 8) + ((r) << 16) + ((a) << 24))
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
int screen_height;
int screen_width;
int screen_bpp;
void *buffer;
void (*pixelGL)(int x, int y, int c); //pointer for pixel function.
bool use_opengl = false; //true if using pointer to pixel function instead of pointer to video memory. (for opengl)
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~SET SCREEN~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
void setScreen(void *pixels, int width, int height, int depth) {
   screen_height = height;
   screen_width = width;
   screen_bpp = depth;
   buffer = pixels;
}

void setScreenGL(void (*pixels)(int, int, int), int width, int height, int depth) {
   screen_height = height;
   screen_width = width;
   screen_bpp = depth;
   pixelGL = pixels;
   use_opengl = true;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~PCX ANGLE DISPLAY~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//Displays an image acording to its bearing: (this needs to actaully change the pcx* store so pixel colision detection is perfect!!!!!!!!!!!!!!!!!) the next function should do that...
void pcxAngleDisplay(int sx, int sy, pcx *store, int angle, int overrideColour) {

   //int t = angle * (M_PI / 180); //convert degres to radians.
   double t = (double) angle / 360 * M_PI * 2;  

   double cosT = cos(t);
   double sinT = sin(t);
   int x1 = store->x / 2;
   int y1 = store->y / 2; //rotation point.

   for (int y = -1 * y1; y < y1; y++) {
      for (int x = -1 * x1; x < x1; x++) {
         int xp = (x * cosT + y * sinT) + x1;
         int yp = (y * cosT - x * sinT) + y1;
         //if (x', y') is in the bounds of the bitmap, 
         //  get pixel(x',y') and plot the pixel to (x,y) on screen.        
         if (xp > 0 && xp < store->x && yp > 0 && yp < store->y) {
            char c = store->d[(yp * store->x) + xp];
            //if (overrideColour != NULL)
            if (overrideColour != 0 && c != CLEAR)
               pixel(sx + x, sy + y, overrideColour);
            if (c != CLEAR)
               pixel(sx + x, sy + y, c);
         }
      }
   }

   //write(sx + 10, sy, angle);

} //END-O-PCX-ANGLE-DISPLAY

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~PCX ANGLE STORE~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//restores an image at a different angle (in a new location):
void pcxRestore(pcx *store, int angle, pcx *newstore) {

   memset(newstore->d, 0, store->x * store->y * sizeof(char));

   //int t = angle * (M_PI / 180); //convert degres to radians.
   double t = (double) angle / 360 * M_PI * 2;  

   double cosT = cos(t);
   double sinT = sin(t);
   int x1 = store->x / 2;
   int y1 = store->y / 2; //rotation point.

   for (int y = -1 * y1; y < y1; y++) {
      for (int x = -1 * x1; x < x1; x++) {
         int xp = (x * cosT + y * sinT) + x1;
         int yp = (y * cosT - x * sinT) + y1;
         //if (x', y') is in the bounds of the bitmap, 
         //  get pixel(x',y') and plot the pixel to (x,y) on screen.        
         if (xp > 0 && xp < store->x && yp > 0 && yp < store->y) {
            char c = store->d[(yp * store->x) + xp];
            if (c != CLEAR) {
               int xx = x + x1; int yy = y + y1;
               if (xx > 0 && xx < store->x && yy > 0 && yy < store->x)
                  newstore->d[(yy * store->x) + xx] = c;
            }
         }
      }
   }

} //END-O-PCX-ANGLE-DISPLAY

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~PCX DISPLAY~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//Displays an image:
void pcxDisplay(int x, int y, pcx *store, int overrideColour) {

   //Declarations:
   char c;

   //display pcx image:
   for (int o = 0; o < store->y; o++) {
      for (int i = 0; i < store->x; i++) {

         c = store->d[(o * store->x) + i]; //get pixel colour.

         //remove transparent colour:
         if (c != CLEAR) {

            //override natural colour of image if requested:
//          if (overrideColour != NULL)
            if (overrideColour != 0)
               pixel(x + i, y + o, overrideColour);
            else
               pixel(x + i, y + o, c);
         }
      }
   }

} //END-O-PCX_DISPLAY

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~PCX LOAD~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//function to read and store a pcx file.
//256 colour only (8-bit), does't read pallette as of yet. (load palette seperatly using getPalette("palette.gpl") )
void pcxStore(char pic[12], pcx *store) {

   //Declarations:
   FILE *picture = fopen(pic, "rb");
   int length;
   int num = 0;
   unsigned char colour;

   //check if file can not be opened:
   if (picture == NULL) { printf("ERROR: Can't open da file '%s'!\n", pic); exit(1); }

   //store x,y values in pic struct:
   fseek(picture, 8, SEEK_SET);
   store->x = readInt(picture) + 1;
   store->y = readInt(picture) + 1;

   //allocate room for bitmap:
   store->d = new unsigned char[store->x * store->y];

   //check if enough memory is availible for storage of picture:
   if (store->d == NULL && store->d32 == NULL) {
      printf("ERROR: No memory available for dis picture (%s), exiting programination...\n", pic);
      exit(1);
   }

   //get bytes/line:
   fseek(picture, 66, SEEK_SET);
   length = readInt(picture);

   //store picture in array:
   fseek(picture, 128, SEEK_SET);
   for (int o = 0; o < store->y; o++) {
      for (int i = 0; i < length; i++) {
         if (num != 0) {
            store->d[(o * store->x) + i] = colour;
            num--;
         }
         else {
            colour = readChar(picture);
            if ((colour & 192) == 192) { // 192 = 11000000 binary
               num = (colour & 63) - 1; // 63 = 00111111 binary
               colour = readChar(picture);
            }
            store->d[(o * store->x) + i] = colour;
         }
      }
   }

   //close file:
   fclose(picture);

} //END-O-PCX-LOAD


//_______________________________________________________________________________________________________________________
//_______________________________16, 24, 32 bit versions of the above three functions are below__________________________
//_______________________________________________________________________________________________________________________
//Displays an image:
void pcxDisplay32(int x, int y, pcx *store, int overrideColour) {

   if (screen_bpp == 8)
      pcxDisplay(x, y, store, overrideColour);
   else {

      //Declarations:
      int c;

      //display pcx image:
      for (int o = 0; o < store->y; o++) {
         for (int i = 0; i < store->x; i++) {

            c = store->d32[(o * store->x) + i]; //get pixel colour.

            //remove transparent colour:
            int black = colour(0, 0, 0);
            if (c != black) {

               //override natural colour of image if requested:
   //          if (overrideColour != NULL)
               if (overrideColour != 0)
                  pixel(x + i, y + o, overrideColour);
               else
                  pixel(x + i, y + o, c);
            }
         }
      }

   }

} //END-O-PCX_DISPLAY
//_______________________________________________________________________________________________________________________
//_______________________________________________________________________________________________________________________
//restores an image at a different angle (in a new location):
void pcxRestore32(pcx *store, int angle, pcx *newstore) {

   newstore->x = store->x;
   newstore->y = store->y;

   if (screen_bpp == 8) {
      newstore->d = new unsigned char[store->x * store->y];
      pcxRestore(store, angle, newstore);
   } else {

      newstore->d32 = new int[store->x * store->y];
      memset(newstore->d32, 0, store->x * store->y * sizeof(int));
   
      //int t = angle * (M_PI / 180); //convert degres to radians.
      double t = (double) angle / 360 * M_PI * 2;  

      double cosT = cos(t);
      double sinT = sin(t);
      int x1 = store->x / 2;
      int y1 = store->y / 2; //rotation point.

      for (int y = -1 * y1; y < y1; y++) {
         for (int x = -1 * x1; x < x1; x++) {
            int xp = (x * cosT + y * sinT) + x1;
            int yp = (y * cosT - x * sinT) + y1;
            //if (x', y') is in the bounds of the bitmap, 
            //  get pixel(x',y') and plot the pixel to (x,y) on screen.        
            if (xp > 0 && xp < store->x && yp > 0 && yp < store->y) {
               int c = store->d32[(yp * store->x) + xp];
               if (c != CLEAR) {
                  int xx = x + x1; int yy = y + y1;
                  if (xx > 0 && xx < store->x && yy > 0 && yy < store->x)
                     newstore->d32[(yy * store->x) + xx] = c;
               }
            }
         }
      }

   }

} //END-O-PCX-ANGLE-DISPLAY
//_______________________________________________________________________________________________________________________
//_______________________________________________________________________________________________________________________
typedef struct {
  char r;
  char g;
  char b;
  char unused;
} SDL_Color;

void pcxStore32(char pic[12], pcx *store) {

   if (screen_bpp == 8)
      pcxStore(pic, store);
   else {
   
      //Declarations:
      FILE *picture = fopen(pic, "rb");
      int length;
      int num = 0;
      unsigned char col;

      //retrieve palete:
      //check for fonts file in several locations:
      char *palettefile = (char *) "/usr/local/share/libpcxf/palette.gpl";
      FILE *palettetest = fopen("/usr/local/share/libpcxf/palette.gpl", "rb");
      if (palettetest == NULL) { 
         palettefile = (char *) "/usr/share/libpcxf/palette.gpl";    
         palettetest = fopen("/usr/share/libpcxf/palette.gpl", "rb");
         if (palettetest == NULL) {
            palettefile = (char *) "palette.gpl"; 
         }
      }
      if (palettetest != NULL) fclose(palettetest);

      SDL_Color *palette = new SDL_Color[256];
      palette = (SDL_Color *) getPalette(palettefile);

      //check if file can not be opened:
      if (picture == NULL) { printf("ERROR32: Can't open da file '%s'!\n", pic); exit(1); }
      //lets make this automatic as i don't plan on doing palette switching...

      //store x,y values in pic struct:
      fseek(picture, 8, SEEK_SET);
      store->x = readInt(picture) + 1;
      store->y = readInt(picture) + 1;

      //allocate room for bitmap:
      store->d32 = new int[store->x * store->y];

      //check if enough memory is availible for storage of picture:
      if (store->d == NULL && store->d32 == NULL) {
         printf("ERROR32: No memory available for dis picture (%s), exiting programination...\n", pic);
         exit(1);
      }

      //get bytes/line:
      fseek(picture, 66, SEEK_SET);
      length = readInt(picture);

      //store picture in array:
      fseek(picture, 128, SEEK_SET);
      for (int o = 0; o < store->y; o++) {
         for (int i = 0; i < length; i++) {
            if (num != 0) {
               SDL_Color colours = palette[col];
               store->d32[(o * store->x) + i] = colour(colours.r, colours.g, colours.b);
               num--;
            }
            else {
               col = readChar(picture);
               if ((col & 192) == 192) { // 192 = 11000000 binary
                  num = (col & 63) - 1; // 63 = 00111111 binary
                  col = readChar(picture);
               }
               SDL_Color colours = palette[col];
               store->d32[(o * store->x) + i] = colour(colours.r, colours.g, colours.b);
            }
         }
      }
      delete palette;
   
   }

}
//_______________________________________________________________________________________________________________________
//_______________________________________________________________________________________________________________________
void drawOn(pcx *store) {
   if (screen_bpp == 8)
      setScreen(store->d, store->x, store->y, 8);
   else
      setScreen(store->d32, store->x, store->y, 32);
}

//_______________________________________________________________________________________________________________________
//_______________________________________________________________________________________________________________________

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~PCX INFO~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
/*
//Function to display the information in the header of a PCX file:
void pcxInfo(char pic[12]) {

   //Declarations:
   FILE *picture = fopen(pic, "rb");
   int ver, bp, np;

   //display Image Information:
   cout << "\n'" << pic << "'";
   fseek(picture, 0, SEEK_SET);
   cout << "\nManufacturer: " << (int) readChar(picture) << "\n";
   ver = (int) readChar(picture);
   cout << "Version: " << ver << "\n";
   cout << "Encoding?: " << (int) readChar(picture) << "\n";
   bp = (int) readChar(picture);
   cout << "Bits/Pixel: " << bp << "\n";

   cout << "Window (Xmin, Ymin, Xmax, Ymax): ";
   fseek(picture, 4, SEEK_SET);
   for (int i = 4; i <= 10; i += 2) {
      fseek(picture, i, SEEK_SET);
      cout << readInt(picture) << ",";
   }

   fseek(picture, 12, SEEK_SET);
   cout << "\nResolution: " << readInt(picture);
   cout << " x " << readInt(picture);                                  

   fseek(picture, 65, SEEK_SET);
   np = readChar(picture);
   cout << "\nNum. Planes: " << np;
   cout << "\nBytes/Line: " << readInt(picture);
   cout << "\nPallette Info: " << readInt(picture) << "\n";

   if ((bp * np) == 24)
      cout << "24-bit image (16M colours).\nNo palette.\n";
   else {
      cout << (bp * np) << "-bit image (" << pow(2, (bp * np)) << " colours).  ";
      if (ver == 5)
         cout << "Contains palette.\n";
   }

   //close file:
   fclose(picture);
}
*/
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~DRAW LINE~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
void line(int x1, int y1, int x2, int y2, int colour) {

   //Declarations:
   double slope;
   int tmp;
   int i;

   //switch line if it's backwards:
   if (x2 < x1) {
      tmp = y1; y1 = y2; y2 = tmp;
      tmp = x1; x1 = x2; x2 = tmp;
   }
   if (x1 == x2 && y2 < y1) {
      tmp = y1; y1 = y2; y2 = tmp;
      tmp = x1; x1 = x2; x2 = tmp;
   }

   //if slope is undefined: (vertical line)
   if (x1 == x2) {
      for (i = 0; i <= abs(y2 - y1); i++) {
         int Y = i + y1;
         pixel(x1, i + y1, colour);
      }
   }

/*
   //if line is horizontal:
   else if (y1 == y2) {

      //speedup for horizontal lines: (this speedup causes a segfault for some reason; thus it is not used)
      if (y1 < screen_height && y1 >= 0) {
         memset((char *) buffer + (y1 * screen_width) + x1, colour, (x2 > screen_width)?(screen_width - x1):(x2 - x1));
      }
   }
*/    
   //if slope is defined:
   else {

      //determine slope:
      slope = (double) (y2 - y1) / (double) (x2 - x1);

      //if line is drawn better with x as base (slope is less than 1):
      if (abs((int) slope) <= 1) {
         for (i = 0; i <= abs(x2 - x1); i++) {
            pixel(i + x1, (int) (slope * i) + y1, colour);
         }
      }

      //if line is drawn better with y as base (slope is greater than 1):
      else {
         //if slope is negative:
         if (slope < 0) {
            for (i = 0; i >= abs(y2 - y1) * -1; i--) {
               pixel(x1 + (int) ((1 / slope) * i), y1 + i, colour);
            }
         }
         //if slope is positive:
         else {
            for (i = 0; i >= abs(y2 - y1) * -1; i--) {
               pixel(x1 - (int) ((1 / slope) * i), y1 - i, colour);
            }
         }
      }
   }

} //end-o-line.

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~DRAW RECTANGLE~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
void rectangle(int x1, int y1, int x2, int y2, int colour) {

   //rectangle draws a filled in block:
   for (int y = y1; y < y2; y++) {
      for (int x = x1; x < x2; x++) {
         pixel(x, y, colour);
      }
   }
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~STORE ARRAY OF IMAGES~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
void pcxStoreArray(char pic[12], pcx *store[], int number, int sizeX, int sizeY) {

   //Declarations:
   pcx temp;

   //load entire file into temporary memory:
   pcxStore(pic, &temp);


   //extract each individual image and store in array of pcx's:
   for (int q  = 0; q < number; q++) {
      store[q]->x = sizeX;
      store[q]->y = sizeY;

      for (int o = 0; o < sizeY; o++) {
         for (int i = 0; i < sizeX; i++) {
            store[q]->d[(o * sizeX) + i] = temp.d[(o * number * sizeX) + ((q * sizeX) + i)];
         }
      }

   }

   //delete temp.d from memory:
   delete temp.d;

} //end-o-pcxStoreArray.

//~~~~~~~~~~~~~~~~~~~~~~~~Load fonts to be used in WRITE function~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
void loadFonts(void) {

   //initilize and check for availible memory for fonts:
   for (int q  = 0; q < 95; q++) {
      if ((letters[q] = new pcx) == NULL || (letters[q]->d = new unsigned char[35]) == NULL) {
         printf("ERROR: Not enough memory for fonts.\n");
         exit(1);
      }
   }

   //check for fonts file in several locations:
   char *fontsfile = (char *) "/usr/local/share/libpcxf/alphabit.pcx";
   FILE *fonttest = fopen("/usr/local/share/libpcxf/alphabit.pcx", "rb");
   if (fonttest == NULL) { 
      fontsfile = (char *) "/usr/share/libpcxf/alphabit.pcx";     
      fonttest = fopen("/usr/share/libpcxf/alphabit.pcx", "rb");
      if (fonttest == NULL) {
         fontsfile = (char *) "alphabit.pcx"; 
      }
   }
   if (fonttest != NULL) fclose(fonttest);

   //store letters in individual pcx sructs:
   pcxStoreArray((char *) fontsfile, (pcx **) letters, 95, 5, 7);

   //report loading of fonts:
   fontsLoaded = 1;

} //end-o-load-fonts.

//~~~~~~~~~~~~~~~~~~~~~~~~remove fonts from memory~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
void unloadFonts(void) {
   for (int q  = 0; q < 95; q++) {
      delete letters[q]->d;
      delete letters[q];
   }
}

//~~~~~~~~~~~~~~~~~~~~~~~~~Write text in graphics mode~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
void writeString(int x, int y, char *text, int colour) { //String.

   //Declarations:
   int i = 0;

   //check if fonts are loaded:
   if (fontsLoaded != 1) {
      printf("ERROR: Fonts not loaded, run 'loadFonts()'.\n");
      exit(1);
   } else {

      //display each character: //32 - space - (15) - numbers(0-9) - (7) - letters - (6) - letters - (4) - 126
      while (i < strlen(text)) {
         //if character is in font set:
         if ((int) text[i] >= 32 && (int) text[i] <= 126) {
            pcxDisplay(x + (i * 6), y, letters[text[i] - 32], colour);

         //if character is not in font set:
         } else
            pcxDisplay(x + (i * 6), y, letters[0], colour); //use space.

         i++;
      }
   }

} //end-o-write-string.

void writeInt(int x, int y, int num, int colour) { //Number.

   //Declarations:
   char stringbuffer[16];

   sprintf(stringbuffer, "%ld", num);
   writeString(x, y, stringbuffer, colour);

} //end-o-write-number.

void writeDouble(int x, int y, double num, int colour) {

   //Declarations:
   char stringbuffer[16];

   gcvt(num, 6, stringbuffer);
   writeString(x, y, stringbuffer, colour);  

} //end-o-write-double.

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
inline void pixel(int x, int y, int colour) {
   if (use_opengl) {
      (*pixelGL)(x, y, colour);
   } else {
      if (screen_bpp == 8)
         pixel8(x, y, (char) colour);
      else if (screen_bpp == 16)
         pixel16(x, y, (short) colour);
      else if (screen_bpp == 24)
         pixel24(x, y, (bit24) colour);
      else if (screen_bpp == 32)
         pixel32(x, y, (long) colour);
   }
}

inline void pixel8(int x, int y, char colour) {
   if (x >= 0 && x < screen_width && y >= 0 && y < screen_height) {
      *((char *) buffer + (y * screen_width) + x) = colour;
   }
}

inline void pixel16(int x, int y, short colour) {
   if (x >= 0 && x < screen_width && y >= 0 && y < screen_height) {
      *((short *) buffer + (y * screen_width) + x) = colour;
   }
}

inline void pixel24(int x, int y, bit24 colour) {
   if (x >= 0 && x < screen_width && y >= 0 && y < screen_height) {
      *((bit24 *) buffer + (y * screen_width) + x) = colour;
   }
}

inline void pixel32(int x, int y, long colour) {
   if (x >= 0 && x < screen_width && y >= 0 && y < screen_height) {
      *((long *) buffer + (y * screen_width) + x) = colour;
   }
} //End-O-pixel to buffer

int colour(unsigned char r, unsigned char g, unsigned char b) {
   if (screen_bpp == 8)
      return r;
   else if (screen_bpp == 16)
      return (((b >> 3) & 31) + (((g >> 2) & 63) << 5) + (((r >> 3) & 31) << 11)); //((b & 31) + ((g & 31) << 5) + ((r & 31) << 10));
   else if (screen_bpp == 24)
      return ((b) + ((g) << 8) + ((r) << 16));
   else if (screen_bpp == 32) {
      return ((b) + ((g) << 8) + ((r) << 16) + ((255) << 24));
   }
}

//~~~~~~Functions to extract/write char's, int's, and double's to/from file's~~~~~~~~~~~~~~~~~~~~~~
char readChar(FILE *fileo) {
   //Declarations:
   char info;

   info = (char) fgetc(fileo);
   return info;
}

int readInt(FILE *fileo) { //readShort actually.
   //Declarations:
   int info;

   info = fgetc(fileo) + (fgetc(fileo) << 8);
   return info;
}

//below has not been tested:
long readLong(FILE *fileo) {
   long info;

   info = fgetc(fileo) + (fgetc(fileo) << 8) + (fgetc(fileo) << 16) + (fgetc(fileo) << 24);
   return info;
}
//______________________________________________________________________________________________


//_______________________________________getPalette_______________________________________
//defined above somwhere...
/*
typedef struct {
  char r;
  char g;
  char b;
  char unused;
} SDL_Color;
*/

int *getPalette(char pal[12]) {
   
   //Declarations:
   SDL_Color *colours = new SDL_Color[256];
   FILE *palette = fopen(pal, "r");

   //check if file can not be opened:
   if (palette == NULL) { printf("ERROR: Can't open da file '%s'!\n", pal); exit(1); }

   char c[50];
   char num[3];
   int i = 0;
   fgets(c, 50, palette);  fgets(c, 50, palette);  fgets(c, 50, palette);  fgets(c, 50, palette); //first 4 lines are unimportant.
   while(fgets(c, 50, palette) != NULL) { 
      memcpy(&num, c, 3);
      colours[i].r = atoi(num);
      memcpy(&num, &(c[4]), 3);
      colours[i].g = atoi(num);
      memcpy(&num, &(c[8]), 3);
      colours[i].b = atoi(num);
      //printf("String: %s (%d,%d,%d)\n", c, colours[i].r, colours[i].g, colours[i].b);
      i++;
    }

//   // Create a display surface with a grayscale palette
//  //SDL_Color colours[256];
//  // Fill colors with color information
//   for(int j=0;j<256;j++){
//     colours[j].r=j;
//     colours[j].g=j;
//     colours[j].b=j;
//   }

     // Set palette
     return (int *) colours;
}
